/***************************************************************************************************
 *
 *  Nexgen SendToWebsite plugin by {LSN}Meindratheal
 *
 *  $CLASS        STWMain
 *  $VERSION      1.00 (12-04-2011 14:59)
 *  $AUTHOR       {LSN}Meindratheal
 *  $DESCRIPTION  Nexgen SendToWebsite plugin.
 *
 **************************************************************************************************/
class STWMain extends NexgenPlugin;

//Copies of the config vars, for faster access
var bool bShowCommandsOnJoin;
var bool bShowHelpMessage;
var byte bAdvertise[8]; //Show to anyone
var byte bHideCmd[8]; //Hide the say/teamsay when used
var string Message[8]; //Help message
var string LongCmd[8]; //Long command string
var string ShortCmd[8]; //Short command string
var string TargetURL[8]; //URL to link to.

//Internal
var int NumSites; //How many websites are in the list.
var int DeathsForWelcome; //How many deaths should a player have for the message to show? For MH.

const Version = "0.50";

/***************************************************************************************************
 *
 *  $DESCRIPTION  Initializes the plugin. Note that if this function returns false the plugin will
 *                be destroyed and is not to be used anywhere.
 *  $RETURN       True if the initialization succeeded, false if it failed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool initialize()
{
	local STWConfig conf;
	// Load settings.
	if (control.bUseExternalConfig)
	{
		conf = spawn(class'STWConfigExt', self);
	}
	else
	{
		conf = spawn(class'STwConfigSys', self);
	}

	ValidateConfig(conf);

	//Find the DeathsForWelcome.
	if(instr(caps(string(level.game)), "MONSTERHUNT") >= 0)
	{
		//Monsterhunt uses deaths as lives, so broadcasting at
		//0 lives will only work on game over :D
		if(bool(Level.Game.GetPropertyText("bUseLives")))
		{
			DeathsForWelcome = int(Level.Game.GetPropertyText("Lives"));
		}
		else
		{
			DeathsForWelcome = 0;
		}
	}
	else
	{
		DeathsForWelcome = 0;
	}
	control.nscLog("[STW]" @ NumSites @ "websites detected!");

	return True;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Handles a potential command message.
 *  $PARAM        sender  PlayerPawn that has send the message in question.
 *  $PARAM        msg     Message send by the player, which could be a command.
 *  $REQUIRE      sender != none
 *  $RETURN       True if the specified message is a command, false if not.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool HandleMsgCommand(PlayerPawn Sender, string Msg)
{
	local int i;
	local NexgenClient C;

	Msg = class'NexgenUtil'.static.trim(Msg);
	C = control.getClient(Sender);

	if(Sender == None || Msg == "" || C == None)
	{
		//Not enough information, or no client
		return False;
	}

	if(Msg ~= "!STWHelp")
	{
		ShowConsoleHelp(C);
		return True;
	}

	//We can't do a switch, that only works with constants.
	for(i = 0; i < NumSites; i++)
	{
		if(Msg ~= LongCmd[i] || Msg ~= ShortCmd[i])
		{
			//It's this one.
			//WebConnect(Sender, TargetURL[i]);
			C.clientCommand("start" @ TargetURL[i]);
			return True;
		}
	}
	//Not a valid STW command.
	return False;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player has sent a mutate call to the server.
 *  $PARAM        mutateString  Mutator specific string (indicates the action to perform).
 *  $PARAM        sender        Player that has send the message.
 *
 **************************************************************************************************/
function Mutate(string MutateString, PlayerPawn Sender)
{
	local NexgenClient C;

	if(left(MutateString, 4) ~= "STW ")
	{
		MutateString = Mid(MutateString, 4);
		if(left(MutateString, 1) != "!")
		{
			MutateString = "!"$MutateString; //Add an exclamation mark if not used in MS.
		}
		HandleMsgCommand(Sender, MutateString);
	}
	else if(MutateString ~= "STWHelp" || MutateString ~= "!STWHelp")
	{
		HandleMsgCommand(Sender, "!STWHelp");
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is sent to player.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        pri       Player replication info of the sending player.
 *  $PARAM        s         The message that is to be send.
 *  $PARAM        type      Type of the message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorTeamMessage(Actor sender, Pawn receiver, PlayerReplicationInfo pri,
                                 coerce string s, name type, optional bool bBeep)
{
	if(HideCmd(s))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is send to player. Spectators that use say (not teamsay)
 *                seem to be calling this function instead of mutatorTeamMessage.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        msg       The message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $PARAM        type      Type of the message that is to be send.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorBroadcastMessage(Actor sender, Pawn receiver, out coerce string msg,
                                      optional bool bBeep, out optional name type)
{
	if(HideCmd(msg))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called whenever a player has joined the game (after its login has been accepted).
 *  $PARAM        newClient  The player that has joined the game.
 *  $REQUIRE      newClient != none
 *  $OVERRIDE
 *
 **************************************************************************************************/
function playerJoined(NexgenClient newClient)
{
	if(newClient.player != None && newClient.player.playerReplicationInfo != None
	   && newClient.player.playerReplicationInfo.Deaths == DeathsForWelcome)
	{
		ShowWelcomeMsg(newClient);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the welcome message to a player. May also broadcast the help.
 *  $PARAM        C    The NexgenClient who has just logged in.
 *  $REQUIRE      C != None
 *
 **************************************************************************************************/
function ShowWelcomeMsg(NexgenClient C)
{
	local int i;
	if(bShowHelpMessage)
	{
		C.showMsg("This server runs STW v"$Version$",");
		C.showMsg("say !STWHelp for a list of available commands.");
	}
	if(bShowCommandsOnJoin)
	{
		for(i = 0; i < ArrayCount(TargetURL); i++)
		{
			if(bAdvertise[i] > 0 && Message[i] != "")
			{
				//Show this one.
				C.showMsg(Message[i]);
			}
		}
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the help commands to a player.
 *  $PARAM        C    The NexgenClient who has just logged in.
 *  $REQUIRE      C != None
 *
 **************************************************************************************************/
function ShowConsoleHelp(NexgenClient C)
{
	local int i;
	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		if(bAdvertise[i] > 0 && Message[i] != "")
		{
			//Show this one.
			C.showMsg(Message[i]);
		}
	}
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks whether a message should be hidden
 *  $RETURN       True if the message should be hidden, false if it should be sent.
 *
 **************************************************************************************************/
function bool HideCmd(string Cmd)
{
	local int idx;
	idx = GetCmdIndex(Cmd);
	return (idx >= 0 && idx < ArrayCount(TargetURL) && bHideCmd[idx] > 0);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Finds the index of a command.
 *  $PARAM        Cmd    The command whose index we need.
 *  $RETURN       The index of the command, or -1 if not found.
 *
 **************************************************************************************************/
function int GetCmdIndex(string Cmd)
{
	local int i;
	local bool bFound;
	while(i < ArrayCount(TargetURL))
	{
		if(LongCmd[i] ~= Cmd || ShortCmd[i] ~= Cmd)
		{
			return i;
		}
		i++;
	}
	//Not found.
	return -1;
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures the configuration is valid.
 *
 **************************************************************************************************/
function ValidateConfig(STWConfig conf)
{
	local int i;

	NumSites = 0;

	//Load non-array data
	bShowCommandsOnJoin = conf.bShowCommandsOnJoin;
	bShowHelpMessage = conf.bShowHelpMessage;

	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		//Load data into the next slot. If the last loop found invalid data, will overwrite that.
		bAdvertise[NumSites] = conf.bAdvertise[i];
		bHideCmd[NumSites] = conf.bHideCmd[i];
		Message[NumSites] = class'NexgenUtil'.static.trim(conf.Message[i]);
		LongCmd[NumSites] = class'NexgenUtil'.static.trim(conf.LongCmd[i]);
		ShortCmd[NumSites] = class'NexgenUtil'.static.trim(conf.ShortCmd[i]);
		TargetURL[NumSites] = ValidateURL(conf.TargetURL[i]);
		if((bAdvertise[NumSites] > 0 && Message[NumSites] == "")|| LongCmd[NumSites] == "" || TargetURL[NumSites] == "")
		{
			//Not enough information.
			//continue;
		}
		else
		{
			if(left(LongCmd[NumSites], 1) != "!")
			{
				LongCmd[NumSites] = "!"$LongCmd[NumSites];
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingLongCmd(NumSites))
			{
				control.nscLog("[STW] Warning: conflicting command:" @ LongCmd[NumSites]);
			}
			if(ShortCmd[NumSites] == "")
			{
				//Take the first letter of the long command.
				ShortCmd[NumSites] = Left(LongCmd[NumSites], 2); //Include !
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingShortCmd(NumSites))
			{
				control.nscLog("[STW] Warning: conflicting command:" @ LongCmd[NumSites]);
			}
			NumSites++;
		}
	}
	//Validated, now copy back and save config.
	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		conf.bAdvertise[i] = bAdvertise[i];
		conf.bHideCmd[i] = bHideCmd[i];
		conf.Message[i] = Message[i];
		conf.LongCmd[i] = LongCmd[i];
		conf.ShortCmd[i] = ShortCmd[i];
		conf.TargetURL[i] = TargetURL[i];
	}
	conf.SaveConfig();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures a URL is valid.
 *  $PARAM        U    The URL to check
 *  $RETURN       The clean URL.
 *
 **************************************************************************************************/
function string ValidateURL(string URL)
{
	local int PIdx;
	local string Protocol, Address;

	if(URL == "")
	{
		return "";
	}

	//Split apart the protocol and the website.
	PIdx = instr(URL, "://");
	if(PIdx == -1)
	{
		//No protocol.
		Protocol = "HTTP";
		Address  = URL;
	}
	else if(PIdx == 0)
	{
		//For some reason, in the format ://www.website.com
		Protocol = "HTTP";
		Address = Mid(URL, 3);
	}
	else
	{
		//In the format HTTP://www.website.com
		Protocol = Left(URL, PIdx);
		Address = Mid(URL, PIdx + 3);
	}
	if(Address == "" || Protocol == "")
	{
		return "";
	}

	return class'NexgenUtil'.static.trim(Protocol $ "://" $ Address);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a long command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingLongCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		if(LongCmd[CmdIdx] ~= LongCmd[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a short command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingShortCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		if(ShortCmd[CmdIdx] ~= ShortCmd[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Default properties block.
 *
 **************************************************************************************************/
defaultproperties
{
    pluginName="Nexgen SendToWebsite"
    pluginAuthor="{LSN}Meindratheal"
    pluginVersion="1.00 build 1"
}