/**************************************************************************************************
 *
 *  NSE. Nexgen Serverside Extensions by {LSN}Meindratheal.
 *
 *  $CLASS        NSEMain
 *  $VERSION      1.12 (14-01-2011 19:13) Revision 1
 *  $AUTHOR       {LSN}Meindratheal
 *  $DESCRIPTION  Nexgen Serverside Extensions. Public version.
 *
 **************************************************************************************************/
class NSEMain extends NexgenPlugin;

var class<NexgenClientLoginHandler> SavedHandler;

var NexgenClient SavedClient;

var bool bForceHandler;

var bool bFoundNXACE;
var bool bFoundTeamSpeak;

const ForceStartMsg = "%1 has forced the game to start.";
const ForceEndMsg = "%1 has forced the game to end.";
const AdminLoginMsg = "%1 has logged in as an administrator.";
const AdminLoginHiddenMsg = "You have logged in as an administrator.";
const AdminLogoutMsg = "%1 has logged out as an administrator.";
const AdminLogoutHiddenMsg = "You have logged out as an administrator.";
const AlreadyAdminMsg = "You are already logged in as an administrator.";
const AllowOneMsg = "%1 has allowed one user into the server.";
const AllowOneJoinedMsg = "%1 has joined the server with a temporary ID.";
const AllowOneFailedMsg = "WARNING: AllowOne failed, as SavedHandler != None";
const AllowAllMsg = "%1 has changed the login handler to %2";
const AllowAllResetMsg = "%1 has reset the login handler.";
const HandlerLockedMsg = "Failed to change login handler as it has been locked.";

const ConnectingToTS = "Connecting to TeamSpeak...";
const TSRCmd = "Mutate TeamSpeak TSR";
const TSBCmd = "Mutate TeamSpeak TSB";
const TSSCmd = "Mutate TeamSpeak TSS";
const TSTCmd = "Mutate TeamSpeak TST";

const TS_Red  = 0;
const TS_Blue = 1;
const TS_Spec = 2;
const TS_All  = 3;


const NSELogInfoTag = "[NSE-INF]";
const FoundNXACEMsg = "NSE has detected that the Nexgen/ACE plugin is running.";
const FoundTSMsg    = "NSE has detected that TeamSpeak is running.";



/***************************************************************************************************
 *
 *  $DESCRIPTION  Initializes the plugin. Note that if this function returns false the plugin will
 *                be destroyed and is not to be used anywhere.
 *  $RETURN       True if the initialization succeeded, false if it failed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool initialize() {

	local Mutator M;

	//Check the ServerActors for NXACE
	if(instr(Caps(ConsoleCommand("get ini:Engine.Engine.GameEngine ServerActors")),"NEXGENACE112") != -1)
	{
		bFoundNXACE = True;
		control.nscLog(NSELogInfoTag @ FoundNXACEMsg);
	}

	//Check the ServerActors for TeamSpeak
	if(instr(Caps(ConsoleCommand("get ini:Engine.Engine.GameEngine ServerActors")),"TEAMSPEAK") != -1)
	{
		bFoundTeamSpeak = True;
		control.nscLog(NSELogInfoTag @ FoundTSMsg);
	}
	//It might be in Mutators. They will have loaded by now!
	else
	{
		for(M = Level.Game.BaseMutator; M != None; M = M.NextMutator)
		{
			if(InStr(Caps(M.Class),"TEAMSPEAK") != -1)
			{
				bFoundTeamSpeak = True;
				control.nscLog(NSELogInfoTag @ FoundTSMsg);
				break;
			}
		}
	}

	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when the game is executing it's first tick.
 *                Used to re-set the server name (e.g. JB changes it.)
 *                Also used to correctly setect UTPure (Nexgen sometimes
 *                doesn't detect it, probably because it searches for the
 *                wrong actor class.
 *
 **************************************************************************************************/
function firstTick()
{
	control.applyConfig();
	DoFixCompat();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check for other mutators that might create compatibility issues with Nexgen.
 *
 **************************************************************************************************/
function DoFixCompat() {
	local Actor a;
	local string actorClass;
	local bool bUTPureEnabled;	

	bUTPureEnabled = control.bUTPureEnabled;

	foreach allActors(class'Actor', a) {
		actorClass = caps(string(a.class));
		if (!bUTPureEnabled && instr(actorClass, ".UTPURE") > 0) {
			bUTPureEnabled = true;
			control.bUTPureEnabled = True;
			control.nscLog(control.lng.format(control.lng.compatibilityModeMsg, "UTPure"));
		}
	}
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Handles a potential command message.
 *  $PARAM        sender  PlayerPawn that has send the message in question.
 *  $PARAM        msg     Message send by the player, which could be a command.
 *  $REQUIRE      sender != none
 *  $RETURN       True if the specified message is a command, false if not.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool handleMsgCommand(PlayerPawn sender, string msg) {
	local string cmd;
	local bool bIsCommand;
	
	cmd = class'NexgenUtil'.static.trim(msg);
	bIsCommand = true;
	switch (cmd) {
		case "!forcestart":
		case "!fs":
			ForceGameStart(sender);
			break;
		case "!forceend":
		case "!end":
		case "!fe":
			ForceGameEnd(sender);
			break;
		case "!adminlogin":
		case "!ali":
			ForceAdminLogin(sender);
			break;
		case "!adminlogout":
		case "!alo":
			ForceAdminLogout(sender);
			break;
		case "!reboot":
		case "!reload":
		case "!crash":
		case "!stop":
			ForceReboot(sender);
			break;
		case "!restart":
			ForceRestart(sender);
			break;
		case "!pause":
			PauseGame(sender);
			break;
		case "!allow":
		case "!allow1":
		case "!allowone":
		case "!allownext":
			AllowOne(sender);
			break;
		case "!stdlogin":
		case "!standardlogin":
		case "!normallogin":
		case "!nohwid":
			SetLoginHandler(sender, class'NexgenClientLoginHandler');
			break;
		case "!allowall":
		case "!nocheck":
		case "!noauth":
		case "!nologin":
			SetLoginHandler(sender, class'NullClientLoginHandler');
			break;
		case "!checklogin":
		case "!dologin":
		case "!doauth":
		case "!check":
		case "!reset":
			ResetLoginHandler(sender);
			break;
		//TS
		case "!tsred":
		case "!tsr":
			ConnectTS(Sender, TS_Red);
			break;
		case "!tsblue":
		case "!tsb":
			ConnectTS(Sender, TS_Blue);
			break;
		case "!tsspectator":
		case "!tsspec":
		case "!tss":
			ConnectTS(Sender, TS_Spec);
			break;
		case "!tsall":
		case "!tstogether":
		case "!tst":
		case "!tsa":
			ConnectTS(Sender, TS_All);
			break;
		// Not a command.
		default: bIsCommand = false;
	}
	
	return bIsCommand;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces the game to start.
 *
 **************************************************************************************************/
function ForceGameStart(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState == control.gInf.GS_Playing || control.gInf.gameState == control.gInf.GS_Ended)
	{
		return;
	}

	//Force the game to start.
	control.gInf.gameState = control.gInf.GS_Playing;
	DeathMatchPlus(level.game).bRequireReady = false;
	DeathMatchPlus(level.game).startMatch();
	control.logAdminAction(Client, ForceStartMsg, Client.playerName);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces the game to end.
 *
 **************************************************************************************************/
function ForceGameEnd(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);
	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState != control.gInf.GS_Playing)
	{
		return;
	}

	//Force the game to end.
	Core.endGame();
	control.logAdminAction(Client, ForceEndMsg, Client.playerName);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a player in as an admin (if allowed).
 *                Hides the standard messages, but the name will still be in
 *                white on the scoreboard - there is no fix besides changing the code.
 *
 **************************************************************************************************/
function ForceAdminLogin(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
	if(client == None || !client.hasRight(client.R_ServerAdmin))
	{
		return;
	}

	if(Sender.bAdmin)
	{
		if(client != None)
		{
			client.ShowMsg(AlreadyAdminMsg);
		}
		else
		{
			Sender.ClientMessage(AlreadyAdminMsg);
		}
	}

	//Login as an administrator.
//	Level.Game.AdminLogin(Sender, control.sConf.decode(control.sConf.CS_GlobalServerSettings, control.sConf.globalAdminPassword));
	//Stealth.
	Sender.PlayerReplicationInfo.bAdmin = True;
	Sender.bAdmin = True;

	if(client != None)
	{
		client.ShowMsg(AdminLoginHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLoginMsg,client.playerName), control.LT_AdminAction);
	}
	else
	{
		Sender.ClientMessage(AdminLoginHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLoginMsg,Sender.PlayerReplicationInfo.playerName), control.LT_AdminAction);
	}
	//Ensure logging to main console
	if(!control.sConf.logToConsole)
	{
		Log(control.lng.adminActionLogTag @ class'NexgenUtil'.static.format(AdminLoginMsg,Sender.PlayerReplicationInfo.playerName),control.logTag);
	}

}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a player out as an admin (if allowed).
 *
 **************************************************************************************************/
function ForceAdminLogout(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);

	if (client == None || !Sender.bAdmin)
	{
		return;
	}

//	Sender.ConsoleCommand("AdminLogout");
	//Stealth
	Sender.PlayerReplicationInfo.bAdmin = False;
	Sender.bAdmin = False;
	if(Sender.ReducedDamageType == 'All')
	{
		Sender.ReducedDamageType = '';
	}
	Sender.StartWalk();

	if(client != None)
	{
		client.ShowMsg(AdminLogoutHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLogoutMsg,client.playerName), control.LT_AdminAction);
	}
	else
	{
		Sender.ClientMessage(AdminLogoutHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLogoutMsg,Sender.PlayerReplicationInfo.playerName), control.LT_AdminAction);
	}
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Reboots the server.
 *
 **************************************************************************************************/
function ForceReboot(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);

	if(client == None)
	{
		return;
	}

	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	//Preliminary checks unnecessary, rights are checked in the rebootServer() call.
	//and no messages are logged in this function.
	Core.RebootServer();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Restarts the current game.
 *
 **************************************************************************************************/
function ForceRestart(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);

	if(client == None)
	{
		return;
	}

	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (!client.hasRight(client.R_MatchAdmin))
	{
		return;
	}

	Core.restartGame();
	//No logging necessary, restartGame() does it.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Pauses the current game.
 *
 **************************************************************************************************/
function PauseGame(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);
	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState != control.gInf.GS_Playing)
	{
		return;
	}

	Core.pauseGame();
	//No logging necessary, restartGame() does it.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces a set login handler.
 *
 **************************************************************************************************/
function SetLoginHandler(PlayerPawn Sender, class<NexgenClientLoginHandler> HandlerClass)
{
	local NexgenClient Client;

	local string HandlerString;

	Client = control.GetClient(Sender);

	// Preliminary checks.
	//If an admin gets locked out, they can login as a ServerAdmin and do this.
	//Allow if: 	if (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))
	if (!(Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))))
	{
		return;
	}

	bForceHandler = True;
	//Only save handler if it isn't already different, prevents multiple calls forgetting the original handler.
	if(SavedHandler == None)
	{
		SavedHandler = control.loginHandler;
	}
	control.loginHandler = HandlerClass;

	HandlerString = String(HandlerClass);
	//Friendly names
	if(HandlerString ~= "NexgenClientsideExtensions112.NullClientLoginHandler")
	{
		HandlerString = "a non-authorising handler.";
	}
	else if(HandlerString ~= (class'NexgenUtil'.default.PackageName $ ".NexgenClientLoginHandler"))
	{
		HandlerString = "the standard handler.";
	}
	else
	{
		HandlerString = Mid(String(HandlerClass),InStr(String(HandlerClass),".") + 1);
	}

	if(client != None)
	{
		control.logAdminAction(Client, AllowAllMsg, Client.playerName, HandlerString);
	}
	else
	{
		//Admin may be unable to login. No client.
		Sender.ClientMessage(class'NexgenUtil'.static.format(AllowAllMsg,"An administrator", HandlerString));
		control.broadcastMsg(AllowAllMsg,"An administrator");
		control.nscLog(class'NexgenUtil'.static.format(AllowAllMsg,"An administrator", HandlerString), control.LT_AdminAction);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Resets the login handler to what it was.
 *
 **************************************************************************************************/
function ResetLoginHandler(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);

	// Preliminary checks.
	//Allow if bForceHandler && (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))
	if(!(bForceHandler && (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))))
	{
		return;
	}

	bForceHandler = False;
	control.loginHandler = SavedHandler;
	SavedHandler = None;
	if(client != None)
	{
		control.logAdminAction(Client, AllowAllResetMsg, Client.playerName);
	}
	else
	{
		//Admin may be unable to login. No client.
		Sender.ClientMessage(class'NexgenUtil'.static.format(AllowAllResetMsg,"An administrator",));
		control.broadcastMsg(AllowAllResetMsg,"An administrator");
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Allows a single player into the game without ID checking.
 *
 **************************************************************************************************/
function AllowOne(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
//	if (client == None || !client.hasRight(client.R_ServerAdmin))
	//If an admin gets locked out, they can login as a ServerAdmin and AllowOne.
	//Allow if: 	if (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))
	if (!(Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))))
	{
		return;
	}



	if(SavedHandler == None)
	{
		SavedHandler = control.loginHandler;
		control.loginHandler = class'NullClientLoginHandler';
		if(client != None)
		{
			control.logAdminAction(Client, AllowOneMsg, Client.playerName);
		}
		else
		{
			//Admin may be unable to login.
			Sender.ClientMessage(class'NexgenUtil'.static.format(AllowOneMsg,"An administrator"));
			control.broadcastMsg(AllowOneMsg,"An administrator");
			control.nscLog(class'NexgenUtil'.static.format(AllowOneMsg,"An administrator"), control.LT_AdminAction);
		}
	}
	else if(bForceHandler)
	{
		//Handler is locked as something else. This does not need logging.
		if(client != None)
		{
			client.showMsg(HandlerLockedMsg);
		}
		else
		{
			//Admin may be unable to login. No client.
			Sender.ClientMessage(HandlerLockedMsg);
		}
	}
	else //SavedHandler != None
	{
		if(client != None)
		{
			client.showMsg(AllowOneFailedMsg);
		}
		else
		{
			//Admin may be unable to login. No client.
			Sender.ClientMessage(AllowOneFailedMsg);
		}
		control.nscLog(AllowOneFailedMsg, control.LT_System);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a new client has been created. Use this function to setup the new
 *                client with your own extensions (in order to support the plugin).
 *  $PARAM        client  The client that was just created.
 *  $REQUIRE      client != none
 *  $OVERRIDE
 *
 **************************************************************************************************/
function clientCreated(NexgenClient client)
{
	local NCEClient xClient;

	xClient = NCEClient(client.addController(class'NCEClient', self));
	//Is this by either AllowOne or another handler override?
	if(SavedHandler != None && bFoundNXACE)
	{
		//Save the client instance...
		SavedClient = client;	
		//Set timer.
		SetTimer(1, False);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called whenever a player has joined the game (after its login has been accepted).
 *  $PARAM        newClient  The player that has joined the game.
 *  $REQUIRE      client != none
 *  $OVERRIDE
 *
 **************************************************************************************************/
function playerJoined(NexgenClient newClient)
{
	if(SavedHandler != None && !bForceHandler)
	{
		//Player joined via AllowOne, NOT through another handler override.
		//Reset the handler...
		control.loginHandler = SavedHandler;
		SavedHandler = None;
		control.nscLog(class'NexgenUtil'.static.format(AllowOneJoinedMsg,newClient.playerName), control.LT_System);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks and executes time-critical events.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function Timer()
{
	//Do we have a saved client?
	if(SavedClient != None && bFoundNXACE)
	{
		//Counteract NXACE login 
		SavedClient.serverID = control.sConf.serverID;
		SavedClient = None;
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Executes a ConnectToTeamSpeak command.
 *
 **************************************************************************************************/
function ConnectTS(PlayerPawn Sender,byte TSType)
{
	local NexgenClient Client;

	if(!bFoundTeamSpeak)
	{
		return;
	}

	Client = control.GetClient(Sender);
	if(Client == None)
	{
		return;
	}

	Client.showMsg(ConnectingToTS);
	
	switch(TSType)
	{
		case TS_Red:
			Client.clientCommand(TSRCmd);
			break;
		case TS_Blue:
			Client.clientCommand(TSBCmd);
			break;
		case TS_Spec:
			Client.clientCommand(TSSCmd);
			break;
		case TS_All:
			Client.clientCommand(TSTCmd);
			break;
	}
}





/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player has sent a mutate call to the server.
 *  $PARAM        mutateString  Mutator specific string (indicates the action to perform).
 *  $PARAM        sender        Player that has send the message.
 *
 **************************************************************************************************/
function mutate(string mutateString, PlayerPawn sender)
{
	//Check if it is a NSE command, and pass it on if necessary.
	if(caps(left(mutateString,4)) == "NSE " && handleMsgCommand(sender, "!" $ Mid(mutateString,4)))
	{
		//Standard NSE command, has been executed.
		return;
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is sent to player.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        pri       Player replication info of the sending player.
 *  $PARAM        s         The message that is to be send.
 *  $PARAM        type      Type of the message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorTeamMessage(Actor sender, Pawn receiver, PlayerReplicationInfo pri,
                                 coerce string s, name type, optional bool bBeep)
{
	if(HideCmd(s))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is send to player. Spectators that use say (not teamsay)
 *                seem to be calling this function instead of mutatorTeamMessage.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        msg       The message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $PARAM        type      Type of the message that is to be send.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorBroadcastMessage(Actor sender, Pawn receiver, out coerce string msg,
                                      optional bool bBeep, out optional name type)
{
	if(HideCmd(msg))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks whether a message should be hidden
 *  $RETURN       True if the message should be hidden, false if it should be sent.
 *
 **************************************************************************************************/
function bool HideCmd(string Cmd)
{
	return	(Cmd ~= "!ali" ||
		Cmd ~= "!alo");
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Default properties block.
 *
 **************************************************************************************************/
defaultproperties
{
    pluginName="Nexgen Serverside Extensions"
    pluginAuthor="{LSN}Meindratheal"
    pluginVersion="1.11 build 12"
}
