/**************************************************************************************************
 *
 *  NSE. Nexgen Serverside Extensions by {LSN}Meindratheal.
 *
 *  $CLASS        NSEMain
 *  $VERSION      1.11 (05-01-2011 18:54)
 *  $AUTHOR       {LSN}Meindratheal
 *  $DESCRIPTION  Nexgen Serverside Extensions. Public version.
 *
 **************************************************************************************************/
class NSEMain extends NexgenPlugin;

var class<NexgenClientLoginHandler> SavedHandler;

var NexgenClient SavedClient;

var bool bForceHandler;

var bool bDoneCheck;

var bool bFoundNXACE;

const ForceStartMsg = "%1 has forced the game to start.";
const ForceEndMsg = "%1 has forced the game to end.";
const AdminLoginMsg = "%1 has logged in as an administrator.";
const AdminLoginHiddenMsg = "You have logged in as an administrator.";
const AdminLogoutMsg = "%1 has logged out as an administrator.";
const AdminLogoutHiddenMsg = "You have logged out as an administrator.";
const AlreadyAdminMsg = "You are already logged in as an administrator.";
const AllowOneMsg = "%1 has allowed one user into the server.";
const AllowOneJoinedMsg = "%1 has joined the server with a temporary ID.";
const AllowOneFailedMsg = "WARNING: AllowOne failed, as SavedHandler != None";
const AllowAllMsg = "%1 has changed the login handler to %2";
const AllowAllResetMsg = "%1 has reset the login handler.";
const HandlerLockedMsg = "Failed to change login handler as it has been locked.";
const NSELogInfoTag = "[NSE-INF]";
const FoundNXACEMsg = "NSE detected that the HWID plugin is running.";



/***************************************************************************************************
 *
 *  $DESCRIPTION  Initializes the plugin. Note that if this function returns false the plugin will
 *                be destroyed and is not to be used anywhere.
 *  $RETURN       True if the initialization succeeded, false if it failed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool initialize() {

	//Delay checl for NXACE, in case it has not yet been created.
	SetTimer(1, False);

	return true;
}



/***************************************************************************************************
 *
 *  $DESCRIPTION  Handles a potential command message.
 *  $PARAM        sender  PlayerPawn that has send the message in question.
 *  $PARAM        msg     Message send by the player, which could be a command.
 *  $REQUIRE      sender != none
 *  $RETURN       True if the specified message is a command, false if not.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool handleMsgCommand(PlayerPawn sender, string msg) {
	local string cmd;
	local bool bIsCommand;
	
	cmd = class'NexgenUtil'.static.trim(msg);
	bIsCommand = true;
	switch (cmd) {
		case "!forcestart":
		case "!fs":
			ForceGameStart(sender);
			break;
		case "!forceend":
		case "!end":
		case "!fe":
			ForceGameEnd(sender);
			break;
		case "!adminlogin":
		case "!ali":
			ForceAdminLogin(sender);
			break;
		case "!adminlogout":
		case "!alo":
			ForceAdminLogout(sender);
			break;
		case "!reboot":
		case "!reload":
		case "!crash":
		case "!stop":
			ForceReboot(sender);
			break;
		case "!restart":
			ForceRestart(sender);
			break;
		case "!pause":
			PauseGame(sender);
			break;
		case "!allow":
		case "!allow1":
		case "!allowone":
		case "!allownext":
			AllowOne(sender);
			break;
		case "!stdlogin":
		case "!standardlogin":
		case "!normallogin":
		case "!nohwid":
			SetLoginHandler(sender, class'NexgenClientLoginHandler');
			break;
		case "!allowall":
		case "!nocheck":
		case "!noauth":
		case "!nologin":
			SetLoginHandler(sender, class'NullClientLoginHandler');
			break;
		case "!checklogin":
		case "!dologin":
		case "!doauth":
		case "!check":
		case "!reset":
			ResetLoginHandler(sender);
			break;	
		// Not a command.
		default: bIsCommand = false;
	}
	
	return bIsCommand;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces the game to start.
 *
 **************************************************************************************************/
function ForceGameStart(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState == control.gInf.GS_Playing || control.gInf.gameState == control.gInf.GS_Ended)
	{
		return;
	}

	//Force the game to start.
	control.gInf.gameState = control.gInf.GS_Playing;
	DeathMatchPlus(level.game).bRequireReady = false;
	DeathMatchPlus(level.game).startMatch();
	control.logAdminAction(Client, ForceStartMsg, Client.playerName);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces the game to end.
 *
 **************************************************************************************************/
function ForceGameEnd(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);
	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState != control.gInf.GS_Playing)
	{
		return;
	}

	//Force the game to end.
	Core.endGame();
	control.logAdminAction(Client, ForceEndMsg, Client.playerName);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a player in as an admin (if allowed).
 *                Hides the standard messages, but the name will still be in
 *                white on the scoreboard - there is no fix besides changing the code.
 *
 **************************************************************************************************/
function ForceAdminLogin(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
	if(client == None || !client.hasRight(client.R_ServerAdmin))
	{
		return;
	}

	if(Sender.bAdmin)
	{
		if(client != None)
		{
			client.ShowMsg(AlreadyAdminMsg);
		}
		else
		{
			Sender.ClientMessage(AlreadyAdminMsg);
		}
	}

	//Login as an administrator.
//	Level.Game.AdminLogin(Sender, control.sConf.decode(control.sConf.CS_GlobalServerSettings, control.sConf.globalAdminPassword));
	//Stealth.
	Sender.PlayerReplicationInfo.bAdmin = True;
	Sender.bAdmin = True;

	if(client != None)
	{
		client.ShowMsg(AdminLoginHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLoginMsg,client.playerName), control.LT_AdminAction);
	}
	else
	{
		Sender.ClientMessage(AdminLoginHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLoginMsg,Sender.PlayerReplicationInfo.playerName), control.LT_AdminAction);
	}
	//Ensure logging to main console
	if(!control.sConf.logToConsole)
	{
		Log(control.lng.adminActionLogTag @ class'NexgenUtil'.static.format(AdminLoginMsg,Sender.PlayerReplicationInfo.playerName),control.logTag);
	}

}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a player out as an admin (if allowed).
 *
 **************************************************************************************************/
function ForceAdminLogout(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);

	if (client == None || !Sender.bAdmin)
	{
		return;
	}

//	Sender.ConsoleCommand("AdminLogout");
	//Stealth
	Sender.PlayerReplicationInfo.bAdmin = False;
	Sender.bAdmin = False;
	if(Sender.ReducedDamageType == 'All')
	{
		Sender.ReducedDamageType = '';
	}
	Sender.StartWalk();

	if(client != None)
	{
		client.ShowMsg(AdminLogoutHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLogoutMsg,client.playerName), control.LT_AdminAction);
	}
	else
	{
		Sender.ClientMessage(AdminLogoutHiddenMsg);
		control.nscLog(class'NexgenUtil'.static.format(AdminLogoutMsg,Sender.PlayerReplicationInfo.playerName), control.LT_AdminAction);
	}
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Reboots the server.
 *
 **************************************************************************************************/
function ForceReboot(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);

	if(client == None)
	{
		return;
	}

	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	//Preliminary checks unnecessary, rights are checked in the rebootServer() call.
	//and no messages are logged in this function.
	Core.RebootServer();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Restarts the current game.
 *
 **************************************************************************************************/
function ForceRestart(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);

	if(client == None)
	{
		return;
	}

	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (!client.hasRight(client.R_MatchAdmin))
	{
		return;
	}

	Core.restartGame();
	//No logging necessary, restartGame() does it.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Pauses the current game.
 *
 **************************************************************************************************/
function PauseGame(PlayerPawn Sender)
{
	local NexgenClient Client;
	local NexgenClientCore Core;

	Client = control.GetClient(Sender);
	Core = NexgenClientCore(client.getController(class'NexgenClientCore'.default.ctrlID));

	// Preliminary checks.
	if (client == None || !client.hasRight(client.R_MatchAdmin) || control.gInf.gameState != control.gInf.GS_Playing)
	{
		return;
	}

	Core.pauseGame();
	//No logging necessary, restartGame() does it.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Forces a set login handler.
 *
 **************************************************************************************************/
function SetLoginHandler(PlayerPawn Sender, class<NexgenClientLoginHandler> HandlerClass)
{
	local NexgenClient Client;

	local string HandlerString;

	Client = control.GetClient(Sender);

	// Preliminary checks.
	//If an admin gets locked out, they can login as a ServerAdmin and do this.
	//Allow if: 	if (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))
	if (!(Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))))
	{
		return;
	}

	bForceHandler = True;
	//Only save handler if it isn't already different, prevents multiple calls forgetting the original handler.
	if(SavedHandler == None)
	{
		SavedHandler = control.loginHandler;
	}
	control.loginHandler = HandlerClass;

	HandlerString = String(HandlerClass);
	//Friendly names
	if(HandlerString ~= "NexgenClientsideExtensions112.NullClientLoginHandler")
	{
		HandlerString = "a non-authorising handler.";
	}
	else if(HandlerString ~= (class'NexgenUtil'.default.PackageName $ ".NexgenClientLoginHandler"))
	{
		HandlerString = "the standard handler.";
	}
	else
	{
		HandlerString = Mid(String(HandlerClass),InStr(String(HandlerClass),".") + 1);
	}

	if(client != None)
	{
		control.logAdminAction(Client, AllowAllMsg, Client.playerName, HandlerString);
	}
	else
	{
		//Admin may be unable to login. No client.
		Sender.ClientMessage(class'NexgenUtil'.static.format(AllowAllMsg,"An administrator", HandlerString));
		control.broadcastMsg(AllowAllMsg,"An administrator");
		control.nscLog(class'NexgenUtil'.static.format(AllowAllMsg,"An administrator", HandlerString), control.LT_AdminAction);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Resets the login handler to what it was.
 *
 **************************************************************************************************/
function ResetLoginHandler(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);

	// Preliminary checks.
	//Allow if bForceHandler && (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))
	if(!(bForceHandler && (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))))
	{
		return;
	}

	bForceHandler = False;
	control.loginHandler = SavedHandler;
	SavedHandler = None;
	if(client != None)
	{
		control.logAdminAction(Client, AllowAllResetMsg, Client.playerName);
	}
	else
	{
		//Admin may be unable to login. No client.
		Sender.ClientMessage(class'NexgenUtil'.static.format(AllowAllResetMsg,"An administrator",));
		control.broadcastMsg(AllowAllResetMsg,"An administrator");
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Allows a single player into the game without ID checking.
 *
 **************************************************************************************************/
function AllowOne(PlayerPawn Sender)
{
	local NexgenClient Client;

	Client = control.GetClient(Sender);
	// Preliminary checks.
//	if (client == None || !client.hasRight(client.R_ServerAdmin))
	//If an admin gets locked out, they can login as a ServerAdmin and AllowOne.
	//Allow if: 	if (Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin)))
	if (!(Sender.bAdmin || (client != None && client.hasRight(client.R_ServerAdmin))))
	{
		return;
	}



	if(SavedHandler == None)
	{
		SavedHandler = control.loginHandler;
		control.loginHandler = class'NullClientLoginHandler';
		if(client != None)
		{
			control.logAdminAction(Client, AllowOneMsg, Client.playerName);
		}
		else
		{
			//Admin may be unable to login.
			Sender.ClientMessage(class'NexgenUtil'.static.format(AllowOneMsg,"An administrator"));
			control.broadcastMsg(AllowOneMsg,"An administrator");
			control.nscLog(class'NexgenUtil'.static.format(AllowOneMsg,"An administrator"), control.LT_AdminAction);
		}
	}
	else if(bForceHandler)
	{
		//Handler is locked as something else. This does not need logging.
		if(client != None)
		{
			client.showMsg(HandlerLockedMsg);
		}
		else
		{
			//Admin may be unable to login. No client.
			Sender.ClientMessage(HandlerLockedMsg);
		}
	}
	else //SavedHandler != None
	{
		if(client != None)
		{
			client.showMsg(AllowOneFailedMsg);
		}
		else
		{
			//Admin may be unable to login. No client.
			Sender.ClientMessage(AllowOneFailedMsg);
		}
		control.nscLog(AllowOneFailedMsg, control.LT_System);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a new client has been created. Use this function to setup the new
 *                client with your own extensions (in order to support the plugin).
 *  $PARAM        client  The client that was just created.
 *  $REQUIRE      client != none
 *  $OVERRIDE
 *
 **************************************************************************************************/
function clientCreated(NexgenClient client)
{
	//Is this by either AllowOne or another handler override?
	if(SavedHandler != None && bFoundNXACE)
	{
		//Save the client instance...
		SavedClient = client;	
		//Set timer.
		SetTimer(1, False);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called whenever a player has joined the game (after its login has been accepted).
 *  $PARAM        newClient  The player that has joined the game.
 *  $REQUIRE      client != none
 *  $OVERRIDE
 *
 **************************************************************************************************/
function playerJoined(NexgenClient newClient)
{
	if(SavedHandler != None && !bForceHandler)
	{
		//Player joined via AllowOne, NOT through another handler override.
		//Reset the handler...
		control.loginHandler = SavedHandler;
		SavedHandler = None;
		control.nscLog(class'NexgenUtil'.static.format(AllowOneJoinedMsg,newClient.playerName), control.LT_System);
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks and executes time-critical events.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function Timer()
{
	//Do we have a saved client?
	if(SavedClient != None && bFoundNXACE)
	{
		//Counteract NXACE login 
		SavedClient.serverID = control.sConf.serverID;
		SavedClient = None;
	}
	//Check for NXACE
	if(!bDoneCheck)
	{
		CheckNXACE();
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks to see if the NexgenACE plugin is active.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function CheckNXACE()
{
	local int i;

	if(bDoneCheck)
	{
		return;
	}
	
	for(i = 0; i < ArrayCount(control.plugins); i++)
	{
		if(control.plugins[i] == None)
		{
			break;
		}
		if(control.plugins[i].pluginName ~= "Nexgen ACE plugin")
		{
			//NXACE is running.
			bFoundNXACE = True;
			Log(NSELogInfoTag @ FoundNXACEMsg, control.logTag);
		}
	}

	bDoneCheck = True;
}





/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player has sent a mutate call to the server.
 *  $PARAM        mutateString  Mutator specific string (indicates the action to perform).
 *  $PARAM        sender        Player that has send the message.
 *
 **************************************************************************************************/
function mutate(string mutateString, PlayerPawn sender)
{
	//Check if it is a NSE command, and pass it on if necessary.
	if(caps(left(mutateString,4)) == "NSE " && handleMsgCommand(sender, "!" $ Mid(mutateString,4)))
	{
		//Standard NSE command, has been executed.
		return;
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is sent to player.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        pri       Player replication info of the sending player.
 *  $PARAM        s         The message that is to be send.
 *  $PARAM        type      Type of the message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorTeamMessage(Actor sender, Pawn receiver, PlayerReplicationInfo pri,
                                 coerce string s, name type, optional bool bBeep)
{
	if(HideCmd(s))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is send to player. Spectators that use say (not teamsay)
 *                seem to be calling this function instead of mutatorTeamMessage.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        msg       The message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $PARAM        type      Type of the message that is to be send.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorBroadcastMessage(Actor sender, Pawn receiver, out coerce string msg,
                                      optional bool bBeep, out optional name type)
{
	if(HideCmd(msg))
	{
		return false;
	}
	return true;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks whether a message should be hidden
 *  $RETURN       True if the message should be hidden, false if it should be sent.
 *
 **************************************************************************************************/
function bool HideCmd(string Cmd)
{
	return	(Cmd ~= "!ali" ||
		Cmd ~= "!alo");
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Default properties block.
 *
 **************************************************************************************************/
defaultproperties
{
    pluginName="Nexgen Serverside Extensions"
    pluginAuthor="{LSN}Meindratheal"
    pluginVersion="1.11 build 12"
}
