class FastTurnDetect expands Mutator
	config( FastTurnDetect );

const VERSION = "20090209";

/* Ini config settings */
var config bool	bUseServerLogFile;
var config bool	bUseCustomLogFile;

var config bool	bCheckBestProgressor;

var config bool	bBan;
var config bool	bKick;
var config bool	bSimulated;

var config int maxYawLimit;
var config int maxPitchLimit;

var config int maxRecordedYaw;
var config int maxRecordedPitch;

/* Internal class variables */
var private bool bInitialized; // For PreBeginPlay()

var private bool bFirstTick;

var private bool bLogStart; // Used when starting custom logger
var private FTDLog logClass;

var private Pawn pawnArray[32]; // Stores references to players

var private int lastYawArray[32];
var private int lastPitchArray[32];

var private int deltaYawArray[32];
var private int deltaPitchArray[32];

var private int maxDeltaYawArray[32];
var private int maxDeltaPitchArray[32];

var private Pawn lastBestProgressor;

function PreBeginPlay()
{
	// Necessary to get round a bug that causes PreBeginPlay() to get called twice
	if( bInitialized ) return; else bInitialized = true;

	log( "Starting FastTurnDetect 1.1" , 'FastTurnDetect' );
	log( "Please contact the author to submit bugs and suggestions" , 'FastTurnDetect' );

	SaveConfig(); // Save ini settings and create ini file in the same go

	if( Level.NetMode != NM_DedicatedServer )
	{
		log( "Current NetMode is" @ Level.NetMode $ ". This mutator is for dedicated servers.", 'FastTurnDetect' );
		log( "Disabling mutator.", 'FastTurnDetect' );

		Disable('Tick');
	}
	else if( isInServerPackages( "FastTurnDetect" ) )
	{
		log( "This is a serverside mod. Please remove it from ServerPackages.", 'FastTurnDetect' );
		log( "Disabling mutator.", 'FastTurnDetect' );

		Disable('Tick');
	}
	else
		Enable('Tick');

	log( "Settings:" , 'FastTurnDetect' );
	log( "bUseServerLogFile is" @ bUseServerLogFile , 'FastTurnDetect');
	log( "bUseCustomLogFile is" @ bUseCustomLogFile , 'FastTurnDetect');
	log( "bCheckBestProgressor is" @ bCheckBestProgressor , 'FastTurnDetect');
	log( "bBan is" @ bBan , 'FastTurnDetect');
	log( "bKick is" @ bKick , 'FastTurnDetect');
	log( "bSimulated is" @ bSimulated , 'FastTurnDetect');
	log( "maxYawLimit is" @ maxYawLimit , 'FastTurnDetect');
	log( "maxPitchLimit is" @ maxPitchLimit , 'FastTurnDetect');
}

simulated function Tick( float DeltaTime )
{
	local int playerID, lastYaw, lastPitch, deltaYaw, deltaPitch, maxDeltaYaw, maxDeltaPitch;
	local string playerName;
	local Pawn p, bestProgressor;

	if( !bFirstTick )
	{
		bFirstTick = true;
		log( "Checks starting" , 'FastTurnDetect' );
	}

	// 20090209: Simplified conditions at Tick start and avoid delta calculation when teleporting or first spawn

	// If game hasn't started or if there are no players
	if( getTimeSinceGameStart() <= 0 || getNumberOfPlayers() <= 0 )
		return; // Exit

	// 20090209: Change while() into for()
	// Cycle through Pawns
	for( p = Level.PawnList; p != none; p = p.nextPawn )
	{
		// 20090209: Check all players or check only best progressor if bCheckBestProgressor is true
		if( !bCheckBestProgressor || ( bCheckBestProgressor && p == getBestProgressor() ) )
		{
			// Pawn is a player (see isValidPlayer)
			// 20090209: Added check for Health (when a player dies, the Pawn is hidden and ViewRotation is modified)
			if( p.Health > 0 && isValidPlayer( p ) )
			{
				playerID = p.PlayerReplicationInfo.PlayerID; // Retrieve playerID to get information from arrays

				// Refer to ModifyPlayer() for an explanation of the purpose of pawnArray
				// Pitch is reset when entering a teleporter
				// Removed pawnArray[ playerID ] == none
				if( p.bJustTeleported || pawnArray[ playerID ] != p )
				{
					lastPitch = p.ViewRotation.Pitch;
					lastYaw = p.ViewRotation.Yaw;
				}
				else
				{
					// Get data from arrays
					lastYaw = lastYawArray[ playerID ];
					lastPitch = lastPitchArray[ playerID ];

					// Calculate difference in Yaw and Pitch since last Tick
					deltaYaw = calcPlayerYawDifference( p.ViewRotation.Yaw, lastYaw ); // Changed 20090209
					deltaPitch = calcPlayerPitchDifference( p.ViewRotation.Pitch, lastPitch ); // Changed 20090209

					// Check if new maximum has been reached
					if( deltaYaw > maxDeltaYaw )
						maxDeltaYaw = deltaYaw;

					if( deltaPitch > maxDeltaPitch )
						maxDeltaPitch = deltaPitch;

					// Check if max values are beyond ini limits
					if( maxDeltaYaw > maxYawLimit || maxDeltaPitch > maxPitchLimit )
					{
						// Used by handlePlayer()
						maxDeltaYawArray[ playerID ] = maxDeltaYaw;
						maxDeltaPitchArray[ playerID ] = maxDeltaPitch;

						// SaveConfig(); // 20090209: Not necessary

						handlePlayer( p );

						maxDeltaYawArray[ playerID ] = 0;
						maxDeltaPitchArray[ playerID ] = 0;
					}

					lastYaw = p.ViewRotation.Yaw;
					lastPitch = p.ViewRotation.Pitch;

					// Save deltas
					deltaYawArray[ playerID ] = deltaYaw;
					deltaPitchArray[ playerID ] = deltaPitch;
				}

				// Save Yaw and Pitch
				lastYawArray[ playerID ] = lastYaw;
				lastPitchArray[ playerID ] = lastPitch;
			}
		}
	}
}

// Added 20090209
function int getNumberOfPlayers()
{
	local int totalNumPlayers;

	if( Level.Game != none )
	{
		totalNumPlayers = DeathMatchPlus(Level.Game).NumPlayers
			+ DeathMatchPlus(Level.Game).NumBots;

		return totalNumPlayers;
	}

	return -1;
}

// Added 20090209
function private int calcPlayerYawDifference( int currentYaw, int previousYaw )
{
	local int yawDifference;

	// Use absolute function as Yaw values can be negative
	// Whether currentYaw is higher or lower than previousYaw doesn't matter since we're interested in how many units separate them
	yawDifference = abs( abs( currentYaw ) - abs( previousYaw ) );

	return yawDifference;
}

// Added 20090209
function private int calcPlayerPitchDifference( int currentPitch, int previousPitch )
{
	local int pitchDifference;

	// Calculate Pitch difference ( max up 18000<-0/65536->49152 max down )
	if( currentPitch > 0 && currentPitch <= 18000 ) // Upper half
	{
		if( previousPitch > 0 && previousPitch <= 18000 ) // Previous pitch in same half
			pitchDifference = abs( currentPitch - previousPitch );
		else if( previousPitch >= 49152 && previousPitch <= 65536 ) // Previous pitch in lower half
			pitchDifference = currentPitch + ( 65536 - previousPitch );
	}
	else if( currentPitch >= 49152 && currentPitch <= 65536 ) // Lower half
	{
		if( previousPitch > 0 && previousPitch <= 18000 ) // Previous pitch in upper half
			pitchDifference = ( 65536 - currentPitch ) + previousPitch;
		else if( previousPitch >= 49152 && previousPitch <= 65536 ) // Previous pitch in same half
			pitchDifference = abs( currentPitch - previousPitch );
	}

	return pitchDifference;
}

// Use ModifyPlayer() to intercept respawns and reset reference values
function ModifyPlayer( Pawn Other )
{
	local int playerID;

	super.ModifyPlayer( Other );

	// Other has just respawned
	// Any code executed here will be "inserted" into the normal course of a game
	// This approach is better than using an indicator variable as this would prove unreliable

	if( isValidPlayer( Other ) )
	{
		playerID = Other.PlayerReplicationInfo.PlayerID;

		// Some weird stuff happens when a player first spawns (probably a yaw change occuring after ModifyPlayer() )
		// This is an attempt to avoid it producing an incorrect deltaYaw value
		// 20090209: Removed "pawnArray[ playerID ] == none"
		if( pawnArray[ playerID ] != Other ) // If this playerID has never been used or was used by a previous player
		{
			//log( Other.PlayerReplicationInfo.PlayerName @ "has spawned for the first time" );
			pawnArray[ playerID ] = Other; // Indicate that this player has spawned for the first time
		}

		lastYawArray[ playerID ] = Other.ViewRotation.Yaw;
		lastPitchArray[ playerID ] = Other.ViewRotation.Pitch;
	}
}

// Needs some commands to be added
// Ask for feedback from users for this
function Mutate( string mutateString, PlayerPawn pp )
{
    local Pawn p;

	Super.Mutate( mutateString, pp );

	if( isValidPlayer( pp ) && pp.bAdmin )
	{
		if( Left( Caps( mutateString ), 9 ) ~= "FTD RESET" ) // Mutate call is for AMP
		{
			reset();

			pp.ClientMessage( "maxDeltaArrays reset" );
		}
	}
}

function bool HandleEndGame()
{
	super.HandleEndGame();

	SaveConfig();

	stopLog();
}

/* Custom functions */

// Convert Rotational Unreal Units to degrees
function float convertRUUsToDegrees( int ruus )
{
	// Cast ruus to float so result is a float to avoid a result of 0.000000
	return ( ( float(ruus) / 65536 ) * 360 );
}

function int getTimeSinceGameStart()
{
	local float time;

	if( Level.Game != none )
	{
		if( Level.Game.GameReplicationInfo.RemainingTime > 0 )
		{
			time = ( TournamentGameReplicationInfo(Level.Game.GameReplicationInfo).TimeLimit * 60 )
						- Level.Game.GameReplicationInfo.RemainingTime;

			return time;
		}
		else
			return Level.Game.GameReplicationInfo.ElapsedTime;
	}

	return -1;
}

function bool isValidPlayer( Pawn p )
{
	if( p != none && p.PlayerReplicationInfo != none && p.bIsPlayer && !p.IsA('Bot') )
		return true;

	return false;
}

function handlePlayer( Pawn p )
{
	local int playerID;
	local string playerName;

	if( isValidPlayer( p ) )
	{
		playerID = p.PlayerReplicationInfo.PlayerID;

		if( bUseServerLogFile )
		{
			log( "** Anomaly detected **", 'FastTurnDetect' );
			log( "Timestamp:" @ getShortAbsoluteTime() , 'FastTurnDetect' );
			log( "Name:" @ p.PlayerReplicationInfo.PlayerName , 'FastTurnDetect' );
			log( "IP:" @ getPlayerIP( p ) , 'FastTurnDetect' );
			log( "Delta yaw:" @ maxDeltaYawArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaYawArray[ playerID ] ) $ " degrees)" , 'FastTurnDetect' );
			log( "Delta pitch:" @ maxDeltaPitchArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaPitchArray[ playerID ] ) $ " degrees)" , 'FastTurnDetect' );
		}

		if( bUseCustomLogFile )
		{
			if( !bLogStart )
				startLog();

			if ( LogClass != none )
			{
				logClass.logEventString( 'FTD' $ ":" @ "Timestamp:" @ getShortAbsoluteTime() );
				logClass.logEventString( 'FTD' $ ":" @ "Name:" @ p.PlayerReplicationInfo.PlayerName );
				logClass.logEventString( 'FTD' $ ":" @ "IP:" @ getPlayerIP( p ) );
				logClass.logEventString( 'FTD' $ ":" @ maxDeltaYawArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaYawArray[ playerID ] ) $ " degrees)" );
				logClass.logEventString( 'FTD' $ ":" @ maxDeltaPitchArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaPitchArray[ playerID ] ) $ " degrees)" );
			}
		}

		playerName = p.PlayerReplicationInfo.PlayerName;

		if( !bSimulated )
		{
			if( bBan )
				banIP( getPlayerIP( p ) );

			if( bKick || bBan )
				kickPlayer( p );
		}
		else
		{
			if( bBan )
				log( playerName @ "has been banned", 'FastTurnDetect' );
			else if( bKick )
				log( playerName @ "has been kicked off the server", 'FastTurnDetect' );
		}
	}
}

function bool banIP( string ipString )
{
	local int i;

	if( ipString == "" )
		return false;

	if( Level.Game != none )
	{
		if ( Level.Game.CheckIPPolicy( ipString ) )
		{
			for ( i=0; i < 50; i++)
			{
				if ( Level.Game.IPPolicies[ i ] == "" ) // Find empty slot
					break;
			}

			if ( i < 50 ) // Hard limit for standard ippolicies
			{
				Level.Game.IPPolicies[i] = "DENY," $ ipString;
				Level.Game.SaveConfig();

				return true;
			}
		}
	}

	return false;
}

function bool kickPlayer( Pawn p )
{
	if( isValidPlayer( p ) )
	{
		p.Destroy();

		if( p == none ) // Kick succesful
			return true;
	}

	return false;
}

function bool isInServerPackages( string serverPackageName )
{
	local string serverPackages;

	if( serverPackageName != "" )
	{
		serverPackages = CAPS( ConsoleCommand( "get ini:Engine.Engine.GameEngine ServerPackages" ) );

		if( InStr( CAPS( serverPackages ) , CAPS( serverPackageName ) ) != -1 )
			return true;
	}

	return false;
}

/* Start log functions */

function startLog()
{
	logClass = spawn( class'FTDLog' );

	if ( logClass != none )
	{
		log("Starting custom log", 'FastTurnDetect' );

		logClass.startLog();
	}

	bLogStart = True;
}

function stopLog()
{
	if ( logClass != None)
	{
		logClass.stopLog();
		logClass.Destroy();
		logClass = None;
	}
}

/* End log functions */

function string getPlayerIP( Pawn p )
{
	local string ip;

	if( isValidPlayer( p ) )
	{
		ip =  PlayerPawn( p ).GetPlayerNetworkAddress();
		ip = left( ip, Instr( ip, ":" ) );

		return ip;
	}

	return "0.0.0.0";
}

function Pawn getBestProgressor()
{
	local float playerFPH, bestFPH;
	local Pawn p, bestProgressor;

	for( p = Level.PawnList; p != none; p = p.nextPawn )
	{
		if( isValidPlayer( p ) )
		{
			playerFPH = getPlayerFPH( p );

			log( "playerFPH" @ playerFPH );

			if( playerFPH > bestFPH )
				bestProgressor = p;
		}
	}

	return bestProgressor;
}

function float getPlayerFPH( Pawn p )
{
	local float fph;

	if( Level.Game != none )
	{
		if( isValidPlayer( p ) )
		{
			// StartTime seems to be only set on a dedicated server
			fph = ( p.KillCount / ( Level.TimeSeconds - p.PlayerReplicationInfo.StartTime ) ) * 3600;

			return fph;
		}
	}

	return 0.0;
}

// Date and time in YYYY.MM.DD-HH.MM.SS format
function string getShortAbsoluteTime()
{
	local string absoluteTime;

	AbsoluteTime = string( Level.Year );

	if( Level.Month < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Month;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Month;

	if( Level.Day < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Day;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Day;

	if( Level.Hour < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Hour;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Hour;

	if( Level.Minute < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Minute;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Minute;

	if( Level.Second < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Second;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Second;

	return absoluteTime;
}

function private reset()
{
	local int i;

	for( i = 0; i < 32; i++ )
	{
		maxDeltaYawArray[ i ] = 0;
		maxDeltaPitchArray[ i ] = 0;
	}
}

defaultproperties
{
	bKick=false
	bBan=false
	bCheckBestProgressor=false
	bSimulated=false
	bUseServerLogFile=false
	bUseCustomLogFile=true
	maxYawLimit=5461
	maxPitchLimit=4000
}
