class FastTurnDetect expands Mutator
	config( FastTurnDetect );

const VERSION = "20090208";

/* Ini config settings */
var config bool	bUseServerLogFile;
var config bool	bUseCustomLogFile;
var config bool	bKick;
var config bool	bBan;
var config bool	bSimulated;

var config int maxYawLimit;
var config int maxPitchLimit;

/* Internal class variables */
var private bool bInitialized; // For PreBeginPlay()

var private bool bLogStart;

var private FTDAlarmLog logClass;

var private Pawn pawnArray[32];

var private int lastYawArray[32];
var private int lastPitchArray[32];

var private int deltaYawArray[32];
var private int deltaPitchArray[32];

var private int maxDeltaYawArray[32];
var private int maxDeltaPitchArray[32];

function PreBeginPlay()
{
	// Necessary to get round a bug that causes PreBeginPlay() to get called twice
	if ( bInitialized )
		return;
	else
		bInitialized = true;

	log( "Starting FastTurnDetect 1.0B" , 'FastTurnDetect' );
	log( "Please contact the author to submit bugs and suggestions" , 'FastTurnDetect' );

	SaveConfig(); // Save ini settings and create ini file in the same go

	if( Level.netMode != NM_DedicatedServer )
	{
		log("This mutator is for dedicated servers", 'FastTurnDetect' );
		log( "Disabling mutator.", 'FastTurnDetect' );

		Disable('Tick');
	}
	else if( isInServerPackages( "FastTurnDetect" ) )
	{
		log( "This is a serverside mod. Please remove it from ServerPackages.", 'FastTurnDetect' );
		log( "Disabling mutator.", 'FastTurnDetect' );

		Disable('Tick');
	}
	else
		Enable('Tick');
}

simulated function Tick( float DeltaTime )
{
	local int playerID, lastYaw, lastPitch, deltaYaw, deltaPitch, maxDeltaYaw, maxDeltaPitch;
	local Pawn p;

	// Check if game has started
	if( getTimeSinceGameStart() == 0 )
		return;

	p = Level.PawnList; // get start of linked list of Pawns

	while( p != none )
	{
		if( isValidPlayer( p ) )
		{
			playerID = p.PlayerReplicationInfo.PlayerID;

			// Get data from arrays
			lastYaw = lastYawArray[ playerID ];
			lastPitch = lastPitchArray[ playerID ];

			if( p.bJustTeleported ) // Pitch is reset when entering a teleporter
				lastPitch = p.ViewRotation.Pitch;

			// When a player dies, the Pawn is hidden and tweened while a carcass is spawned in it's place
			if( p.GetStateName() == 'Dying' || p.bHidden )
			{
				lastYaw = p.ViewRotation.Yaw;
				lastPitch = p.ViewRotation.Pitch;
			}

			// Refer to ModifyPlayer() for an explanation of why this test is carried out
			if( pawnArray[ playerID ] == none || pawnArray[ playerID ] != p )
				lastYaw = p.ViewRotation.Yaw;

			// Use absolute function as Yaw values can be negative
			deltaYaw = abs( abs( p.ViewRotation.Yaw ) -  abs( lastYaw ) );

			// Calculate Pitch difference ( max up 18000<-0/65536->49152 max down )
			if( p.ViewRotation.Pitch > 0 && p.ViewRotation.Pitch <= 18000 )
			{
					if( lastPitch > 0 && lastPitch <= 18000 )
						deltaPitch = abs( p.ViewRotation.Pitch - lastPitch );
					else if( lastPitch >= 49152 && lastPitch <= 65536 )
						deltaPitch = p.ViewRotation.Pitch + ( 65536 - lastPitch );

				else if( p.ViewRotation.Pitch >= 49152 && p.ViewRotation.Pitch <= 65536 )
				{
					if( lastPitch > 0 && lastPitch <= 18000 )
						deltaPitch = ( 65536 - p.ViewRotation.Pitch ) + lastPitch;
					else if( lastPitch >= 49152 && lastPitch <= 65536 )
						deltaPitch = abs( p.ViewRotation.Pitch - lastPitch );
				}

				if( deltaYaw > maxDeltaYaw )
					maxDeltaYaw = deltaYaw;

				if( deltaPitch > maxDeltaPitch )
					maxDeltaPitch = deltaPitch;

				if( maxDeltaYaw > maxYawLimit || maxDeltaPitch > maxPitchLimit )
				{
					maxDeltaYawArray[ playerID ] = maxDeltaYaw;
					maxDeltaPitchArray[ playerID ] = maxDeltaPitch;

					SaveConfig();

					handlePlayer( p );
				}
			}

			lastYaw = p.ViewRotation.Yaw;
			lastPitch = p.ViewRotation.Pitch;

			// Save data to arrays
			lastYawArray[ playerID ] = lastYaw;
			lastPitchArray[ playerID ] = lastPitch;
			deltaYawArray[ playerID ] = deltaYaw;
			deltaPitchArray[ playerID ] = deltaPitch;
		}

		p = p.nextPawn;
	}
}

// Use ModifyPlayer() to intercept respawns and reset reference values
function ModifyPlayer( Pawn Other )
{
	local int playerID;

	super.ModifyPlayer( Other );

	// Other has just respawned
	// Any code executed here will be "inserted" into the normal course of a game
	// This approach is better than using an indicator variable as this would prove unreliable

	if( isValidPlayer( Other ) )
	{
		playerID = Other.PlayerReplicationInfo.PlayerID;

		// Some weird stuff happens when a player first spawns (probably a yaw change occuring after ModifyPlayer() )
		// This is an attempt to avoid it producing an incorrect deltaYaw value
		if( pawnArray[ playerID ] == none || pawnArray[ playerID ] != Other )
			pawnArray[ playerID ] = Other;

		lastYawArray[ playerID ] = Other.ViewRotation.Yaw;
		lastPitchArray[ playerID ] = Other.ViewRotation.Pitch;
	}
}

function Mutate( string mutateString, PlayerPawn pp )
{
    local Pawn p;

	Super.Mutate( mutateString, pp );

	if( isValidPlayer( pp ) && pp.bAdmin )
	{
		if( Left( Caps( mutateString ), 9 ) ~= "FTD RESET" ) // Mutate call is for AMP
		{
			reset();

			pp.ClientMessage( "maxDeltaArrays reset" );
		}
	}
}

function bool HandleEndGame()
{
	super.HandleEndGame();

	SaveConfig();

	stopLog();
}

/* Custom functions */

// Convert Rotational Unreal Units to degrees
function float convertRUUsToDegrees( int ruus )
{
	// Cast ruus to float so result is a float to avoid a result of 0.000000
	return ( ( float(ruus) / 65536 ) * 360 );
}

function int getTimeSinceGameStart()
{
	local float time;

	if( Level.Game != none )
	{
		if( Level.Game.GameReplicationInfo.RemainingTime > 0 )
		{
			time = ( TournamentGameReplicationInfo(Level.Game.GameReplicationInfo).TimeLimit * 60 )
						- Level.Game.GameReplicationInfo.RemainingTime;

			return time;
		}
		else
			return Level.Game.GameReplicationInfo.ElapsedTime;
	}

	return -1;
}

function bool isValidPlayer( Pawn p )
{
	if( p != none && p.PlayerReplicationInfo != none && p.bIsPlayer && !p.IsA('Bot') )
		return true;

	return false;
}

function handlePlayer( Pawn p )
{
	local int playerID;
	local string playerName;

	if( isValidPlayer( p ) )
	{
		playerID = p.PlayerReplicationInfo.PlayerID;

		if( bUseServerLogFile )
		{
			log( "** Anomaly detected **", 'FastTurnDetect' );
			log( "Timestamp:" @ getShortAbsoluteTime() , 'FastTurnDetect' );
			log( "Name:" @ p.PlayerReplicationInfo.PlayerName );
			log( "IP:" @ getPlayerIP( p ) );
			log( "Delta yaw:" @ maxDeltaYawArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaYawArray[ playerID ] ) $ " degrees)" );
			log( "Delta pitch:" @ maxDeltaPitchArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaPitchArray[ playerID ] ) $ " degrees)" );
		}

		if( bUseCustomLogFile )
		{
			if( !bLogStart )
				startLog();

			if ( LogClass != none )
			{
				logClass.logEventString( 'FTD' $ ":" @ "Timestamp:" @ getShortAbsoluteTime() );
				logClass.logEventString( 'FTD' $ ":" @ "Name:" @ p.PlayerReplicationInfo.PlayerName );
				logClass.logEventString( 'FTD' $ ":" @ "IP:" @ getPlayerIP( p ) );
				logClass.logEventString( 'FTD' $ ":" @ maxDeltaYawArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaYawArray[ playerID ] ) $ " degrees)" );
				logClass.logEventString( 'FTD' $ ":" @ maxDeltaPitchArray[ playerID ] @ "(" $ convertRUUsToDegrees( maxDeltaPitchArray[ playerID ] ) $ " degrees)" );
			}
		}

		playerName = p.PlayerReplicationInfo.PlayerName;

		if( !bSimulated )
		{
			if( bBan )
				banIP( getPlayerIP( p ) );

			if( bKick || bBan )
				kickPlayer( p );
		}

		if( bBan )
			log( playerName @ "has been banned", 'FastTurnDetect' );
		else if( bKick )
			log( playerName @ "has been kicked off the server", 'FastTurnDetect' );
	}
}

function bool banIP( string ipString )
{
	local int i;

	if( ipString == "" )
		return false;

	if( Level.Game != none )
	{
		if ( Level.Game.CheckIPPolicy( ipString ) )
		{
			for ( i=0; i < 50; i++)
			{
				if ( Level.Game.IPPolicies[ i ] == "" ) // Find empty slot
					break;
			}

			if ( i < 50 ) // Hard limit for standard ippolicies
			{
				Level.Game.IPPolicies[i] = "DENY," $ ipString;
				Level.Game.SaveConfig();

				return true;
			}
		}
	}

	return false;
}

function bool kickPlayer( Pawn p )
{
	if( isValidPlayer( p ) )
	{
		p.Destroy();

		if( p == none ) // Kick succesful
			return true;
	}

	return false;
}

function bool isInServerPackages( string serverPackageName )
{
	local string serverPackages;

	if( serverPackageName != "" )
	{
		serverPackages = CAPS( ConsoleCommand( "get ini:Engine.Engine.GameEngine ServerPackages" ) );

		if( InStr( CAPS( serverPackages ) , CAPS( serverPackageName ) ) != -1 )
			return true;
	}

	return false;
}

/* Log functions */

function startLog()
{
	logClass = spawn( class'FTDAlarmLog' );

	if ( logClass != none )
	{
		log("Starting custom log", 'FastTurnDetect' );

		logClass.startLog();
	}

	bLogStart = True;
}

function stopLog()
{
	if ( logClass != None)
	{
		logClass.stopLog();
		logClass.Destroy();
		logClass = None;
	}
}

/* /Log functions */

function string getPlayerIP( Pawn p )
{
	local string ip;

	if( isValidPlayer( p ) )
	{
		ip =  PlayerPawn( p ).GetPlayerNetworkAddress();
		ip = left( ip, Instr( ip, ":" ) );

		return ip;
	}

	return "0.0.0.0";
}

// Date and time in YYYY.MM.DD-HH.MM.SS format
function string getShortAbsoluteTime()
{
	local string absoluteTime;

	AbsoluteTime = string( Level.Year );

	if( Level.Month < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Month;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Month;

	if( Level.Day < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Day;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Day;

	if( Level.Hour < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Hour;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Hour;

	if( Level.Minute < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Minute;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Minute;

	if( Level.Second < 10 )
		AbsoluteTime = AbsoluteTime$".0"$Level.Second;
	else
		AbsoluteTime = AbsoluteTime$"."$Level.Second;

	return absoluteTime;
}

function private reset()
{
	local int i;

	for( i = 0; i < 32; i++ )
	{
		maxDeltaYawArray[ i ] = 0;
		maxDeltaPitchArray[ i ] = 0;
	}
}

defaultproperties
{
	bKick=false
	bBan=false
	bSimulated=false
	bUseServerLogFile=false
	bUseCustomLogFile=true
	maxYawLimit=5461
	maxPitchLimit=4000
}
