//Mutator
class STWMut extends Mutator config(system);

//Global
var config bool bShowCommandsOnJoin;
var config bool bShowHelpMessage;

var config byte bAdvertise[8]; //Show to anyone
var config byte bHideCmd[8]; //Hide the say/teamsay when used
var config string Message[8]; //Help message
var config string LongCmd[8]; //Long command string
var config string ShortCmd[8]; //Short command string
var config string TargetURL[8]; //URL to link to.

//Internal
var int NumSites; //How many websites are in the list.
var int DeathsForWelcome; //How many deaths should a player have for the message to show? For MH.

var STWClient ClientList; //First remote client instance.

var string Version;

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called as soon as this actor is created.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function PreBeginPlay()
{
	ValidateConfig();
	Level.Game.RegisterMessageMutator(self);
	//Find the DeathsForWelcome.
	if(instr(caps(string(level.game)), "MONSTERHUNT") >= 0)
	{
		//Monsterhunt uses deaths as lives, so broadcasting at
		//0 lives will only work on game over :D
		if(bool(Level.Game.GetPropertyText("bUseLives")))
		{
			DeathsForWelcome = int(Level.Game.GetPropertyText("Lives"));
		}
		else
		{
			DeathsForWelcome = 0;
		}
	}
	else
	{
		DeathsForWelcome = 0;
	}
	Log("************************");
	Log("*  SendToWebsite v"$Version$"  *");
	Log("*      is active!      *");
	Log("* "$NumSites$" websites detected! *");
	Log("************************");
	SaveConfig();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Handles a potential command message.
 *  $PARAM        sender     PlayerPawn that has send the message in question.
 *  $PARAM        msg        Message send by the player, which could be a command.
 *  $REQUIRE      sender != none
 *
 **************************************************************************************************/
function HandleMsgCommand(PlayerPawn Sender, string Msg)
{
	local int i;
	Msg = class'STWUtils'.static.trim(Msg);

	if(Sender == None || Msg == "")
	{
		//Not enough information.
		return;
	}

	//We can't do a switch, that only works with constants.
	for(i = 0; i < NumSites; i++)
	{
		if(Msg ~= LongCmd[i] || Msg ~= ShortCmd[i])
		{
			//It's this one.
			WebConnect(Sender, TargetURL[i]);
			return;
		}
	}
	//Not a valid STW command.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player has sent a mutate call to the server.
 *  $PARAM        mutateString  Mutator specific string (indicates the action to perform).
 *  $PARAM        sender        Player that has send the message.
 *
 **************************************************************************************************/
function Mutate(string MutateString, PlayerPawn Sender)
{
	local string Msg;

	Msg = MutateString;

	if(left(Msg, 4) ~= "STW ")
	{
		Msg = Mid(Msg, 4);
		if(left(Msg, 1) != "!")
		{
			Msg = "!"$Msg; //Add an exclamation mark if not used in MS.
		}
		HandleMsgCommand(Sender, Msg);
	}
	else if(Msg ~= "STWHelp" || Msg ~= "!STWHelp")
	{
		ShowConsoleHelp(Sender);
	}

	Super.Mutate(MutateString, Sender);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is sent to player.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        pri       Player replication info of the sending player.
 *  $PARAM        s         The message that is to be send.
 *  $PARAM        type      Type of the message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorTeamMessage(Actor sender, Pawn receiver, PlayerReplicationInfo pri,
                                 coerce string s, name type, optional bool bBeep)
{
	local bool bHideMsg;

	// Check for commands.
	if (sender != none && sender.isA('PlayerPawn') && sender == receiver &&
	    (type == 'Say' || type == 'TeamSay'))
	{
		if(s ~= "STWHelp" || s ~= "!STWHelp")
		{
			ShowConsoleHelp(PlayerPawn(Sender));
		}
		else
		{
			HandleMsgCommand(PlayerPawn(sender), s);
		}
		bHideMsg = HideCmd(s);
	}

	// Allow other message mutators to do their job.
	if (nextMessageMutator != none)
	{
		return	(!bHideMsg &&
			nextMessageMutator.mutatorTeamMessage(sender, receiver, pri, s, type, bBeep));
	}
	return !bHideMsg;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is send to player. Spectators that use say (not teamsay)
 *                seem to be calling this function instead of mutatorTeamMessage.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        msg       The message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $PARAM        type      Type of the message that is to be send.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorBroadcastMessage(Actor sender, Pawn receiver, out coerce string msg,
                                      optional bool bBeep, out optional name type)
{
	local bool bIsSpecMessage, bHideMsg;
	local PlayerReplicationInfo senderPRI;

	// Get sender player replication info.
	if (sender != none && sender.isA('Pawn')) {
		senderPRI = Pawn(sender).playerReplicationInfo;
	}

	// Check if we're dealing with a spectator chat message.
	bIsSpecMessage = senderPRI != none && sender.isA('Spectator') &&
	                 left(msg, len(senderPRI.playerName) + 1) ~= (senderPRI.playerName $ ":");

	// Check for commands.
	if (bIsSpecMessage && sender == receiver)
	{
		if(Msg ~= "STWHelp" || Msg ~= "!STWHelp")
		{
			ShowConsoleHelp(PlayerPawn(Sender));
		}
		else
		{
			HandleMsgCommand(PlayerPawn(sender), mid(msg, len(senderPRI.playerName) + 1));
		}
		bHideMsg = HideCmd(mid(msg, len(senderPRI.playerName) + 1));
	}

	// Allow other message mutators to do their job.
	if (nextMessageMutator != none)
	{
		return	(!bHideMsg &&
			nextMessageMutator.mutatorBroadcastMessage(sender, receiver, msg, bBeep, type));
	}
	return !bHideMsg;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player (re)spawns and allows us to modify the player.
 *  $PARAM        other  The pawn/player that has (re)spawned.
 *  $REQUIRE      other != none
 *  $WARNING      May not be compatible with MonsterHunt.
 *
 **************************************************************************************************/
function ModifyPlayer(Pawn Other) {
	if(Other.IsA('PlayerPawn') && Other.PlayerReplicationInfo.Deaths == DeathsForWelcome)
	{
		ShowWelcomeMsg(PlayerPawn(Other));
	}
	Super.ModifyPlayer(Other);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the welcome message to a player. May also broadcast the help.
 *
 **************************************************************************************************/
function ShowWelcomeMsg(PlayerPawn P)
{
	local int i;
	if(bShowHelpMessage)
	{
		P.ClientMessage("This server runs STW v"$Version$",");
		P.ClientMessage("say !STWHelp for a list of available commands.");
	}
	if(bShowCommandsOnJoin)
	{
		for(i = 0; i < ArrayCount(TargetURL); i++)
		{
			if(bAdvertise[i] > 0 && Message[i] != "")
			{
				//Show this one.
				P.ClientMessage(Message[i]);
			}
		}
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the help commands to a player.
 *
 **************************************************************************************************/
function ShowConsoleHelp(PlayerPawn P)
{
	local int i;
	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		if(bAdvertise[i] > 0 && Message[i] != "")
		{
			//Show this one.
			P.ClientMessage(Message[i]);
		}
	}
}


/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks whether a message should be hidden
 *  $RETURN       True if the message should be hidden, false if it should be sent.
 *
 **************************************************************************************************/
function bool HideCmd(string Cmd)
{
	local int idx;
	idx = GetCmdIndex(Cmd);
	return (idx >= 0 && idx < ArrayCount(TargetURL) && bHideCmd[idx] > 0);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Finds the index of a command.
 *  $PARAM        Cmd    The command whose index we need.
 *  $RETURN       The index of the command, or -1 if not found.
 *
 **************************************************************************************************/
function int GetCmdIndex(string Cmd)
{
	local int i;
	local bool bFound;
	while(i < ArrayCount(TargetURL))
	{
		if(LongCmd[i] ~= Cmd || ShortCmd[i] ~= Cmd)
		{
			return i;
		}
		i++;
	}
	//Not found.
	return -1;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Sends the command to the client. Not necessarily efficient,
 *                but I suck at replication issues.
 *  $PARAM        Sender     PlayerPawn that has called the command
 *  $PARAM        URL        Target URL
 *  $REQUIRE      sender != none
 *
 **************************************************************************************************/
function WebConnect(PlayerPawn Sender, string URL)
{
	local STWClient C, Target;
	for(C = ClientList; C != None; C = C.NextClient)
	{
		if(C.Owner == Sender)
		{
			Target = C;
			break;
		}
	}
	if(Target == None)
	{
		Target = Spawn(class'STWClient', Sender);
		Target.Link(self);
	}
	Target.ClientWebConnect(URL);
}



/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures the configuration is valid.
 *
 **************************************************************************************************/
function ValidateConfig()
{
	local int Loops;

	While(NumSites < ArrayCount(TargetURL) && Loops < 8)
	{
		//Check this website.
		//Trim
		Message[NumSites] = class'STWUtils'.static.trim(Message[NumSites]);
		LongCmd[NumSites] = class'STWUtils'.static.trim(LongCmd[NumSites]);
		ShortCmd[NumSites] = class'STWUtils'.static.trim(ShortCmd[NumSites]);
		TargetURL[NumSites] = ValidateURL(TargetURL[NumSites]);

		if(Message[NumSites] == "" || LongCmd[NumSites] == "" || TargetURL[NumSites] == "")
		{
			//Not enough information.
			ShiftUp(NumSites); //Remove this set from the array.
		}
		else
		{
			if(left(LongCmd[NumSites], 1) != "!")
			{
				LongCmd[NumSites] = "!"$LongCmd[NumSites];
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingLongCmd(NumSites))
			{
				MLog("Warning: conflicting command:" @ LongCmd[NumSites]);
			}
			if(ShortCmd[NumSites] == "")
			{
				//Take the first letter of the long command.
				ShortCmd[NumSites] = Left(LongCmd[NumSites], 2); //Include !
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingShortCmd(NumSites))
			{
				MLog("Warning: conflicting command:" @ ShortCmd[NumSites]);
			}
		}
		NumSites++;
		Loops++; //We should only ever loop 8 times. Without this, we would infinitely loop on blanks.
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures a URL is valid.
 *  $PARAM        U    The URL to check
 *  $RETURN       The clean URL.
 *
 **************************************************************************************************/
function string ValidateURL(string U)
{
	if(instr(U, "http://") == -1)
	{
		//Assume HTTP
		U = "HTTP://"$U;
	}
	//Add further validation here.
	return class'STWUtils'.static.trim(U);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a long command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingLongCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		if(LongCmd[CmdIdx] ~= LongCmd[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a short command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingShortCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		if(ShortCmd[CmdIdx] ~= ShortCmd[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Moves all array elements upwards by one.
 *  $PARAM        Idx   The index to start at.
 *
 **************************************************************************************************/
//Shifts up all array elements below this one.
function ShiftUp(int Idx)
{

	//Index 0 will need 7 shifts
	//Index 2 will need 5 etc

	While(Idx < ArrayCount(TargetURL) - 1)
	{
		bAdvertise[Idx] = bAdvertise[Idx + 1];
		bHideCmd[Idx] = bHideCmd[Idx + 1];
		Message[Idx] = Message[Idx + 1];
		LongCmd[Idx] = LongCmd[Idx + 1];
		ShortCmd[Idx] = ShortCmd[Idx + 1];
		TargetURL[Idx] = TargetURL[Idx + 1];
		Idx++;
	}
	//Idx should now be 7.
	bAdvertise[Idx] = 0;
	bHideCmd[Idx] = 0;
	Message[Idx] = "";
	LongCmd[Idx] = "";
	ShortCmd[Idx] = "";
	TargetURL[Idx] = "";

	NumSites--; //Force a recheck of this element, as it has changed.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a message.
 *  $PARAM        M    The string to log.
 *
 **************************************************************************************************/
function MLog(coerce string M)
{
	Log(M, 'STWMut');
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Default properties block.
 *
 **************************************************************************************************/
defaultproperties
{
  bShowCommandsOnJoin=True
  bShowHelpMessage=True

  Version="0.5"
}