/**
\file d3d.h
*/

#pragma once
#include <D3D10.h>
#include <D3DX10.h>
#include "texturecache.h"

class D3D
{
private:
	static int createRenderTargetViews();
	static int findAALevel();	
	static ID3D10EffectPass* switchToPass(int index);
	
	
public:

	/**
	Projection modes. 
	PROJ_NORMAL is normal projection.
	PROJ_Z_ONLY only applies the projection to the Z coordinate.
	PROJ_COMPENSATE_Z_NEAR projects vertices but adjusts their W value to compensate for being moved inside zNear
	*/
	enum ProjectionMode {PROJ_NORMAL,PROJ_Z_ONLY,PROJ_COMPENSATE_Z_NEAR};

	/**
	Custom flags to set render state from renderer interface.
	*/
	//enum Polyflag {};

	/** 2 float vector */
	struct Vec2
	{
		float x,y;
	};

	/** 3 float vector */
	struct Vec3
	{
		float x,y,z;
	};

	/** 4 float vector */
	struct Vec4
	{
		float x,y,z,w;
	};

	/** 4 byte vector */
	struct Vec4_byte
	{
		BYTE x,y,z,w;
	};

	/** Vertex */
	struct Vertex
	{
		Vec3 Pos;
		Vec4 Color;
		Vec4 Fog;
		Vec3 Normal;
		Vec2 TexCoord[5];
		DWORD flags;
	};

	/** Most basic vertex for post processing */
	struct SimpleVertex
	{
		Vec3 Pos;		
	};



	/** Options, some user configurable */
	static struct Options
	{
		int samples; /**< Number of MSAA samples */
		int VSync; /**< VSync on/off */
		int aniso; /**< Anisotropic filtering levels */
		int LODBias; /**< Mipmap LOD bias */
		float brightness; /**< Game brightness */
		int POM; /**< Parallax occlusion mapping */
		int bumpMapping; /**<Bumpmapping */
		int alphaToCoverage; /**< Alpha to coverage support */
		float zNear; /**< Near Z value used in shader and for projection matrix */
		int classicLighting; /**< Lighting that matches old renderers */
	};

	/**
Shader side variables
*/
static struct ShaderVars
{
	ID3D10EffectMatrixVariable* projection; /**< projection matrix */
	ID3D10EffectScalarVariable* projectionMode; /**< Projection transform mode (near/far) */
	ID3D10EffectScalarVariable* useTexturePass; /**< Bool whether to use each texture pass (shader side) */
	ID3D10EffectShaderResourceVariable* shaderTextures; /**< GPU side currently bound textures */
	ID3D10EffectVectorVariable* flashColor; /**< Flash color */
	ID3D10EffectScalarVariable* flashEnable; /**< Flash enabled? */
	ID3D10EffectScalarVariable* time; /**< Time for sin() etc */
	ID3D10EffectScalarVariable* viewportHeight; /**< Viewport height in pixels */
	ID3D10EffectScalarVariable* viewportWidth; /**< Viewport width in pixels */
	ID3D10EffectScalarVariable* brightness; /**< Brightness 0-1 */
	ID3D10EffectVectorVariable* fogColor; /**< Fog color */
	ID3D10EffectScalarVariable* fogDist; /**< Fog end distance? */
}  shaderVars;
	
	/**@name API initialization/upkeep */
	//@{
	static int init(HWND hwnd,D3D::Options &createOptions);
	static void uninit();
	static int resize(int X, int Y, bool fullScreen);
	//@}
	
	/**@name Setup/clear frame */
	//@{
	static void newFrame();
	static void clear(D3D::Vec4& clearColor);
	static void clearDepth();
	//@}

	/**@name Prepare and render buffers */
	//@{
	static void map(bool clear);
	static void render();
	static void present();
	static void commit();
	//@}

	/**@name Index and buffer vertices */
	//@{
	static void indexTriangleFan(int num);
	static void indexQuad();
	static D3D::Vertex* getVertex();
	
	//@}
	
	/**@name Set state (projection, flags) */
	//@{
	static void setViewPort(int X, int Y, int left, int top);
	static void setProjectionMode(D3D::ProjectionMode mode);
	static void setProjection(float aspect, float XoverZ);
	static void setFlags(int flags, int d3dflags);
	//@}
		
	/**@name Misc */
	//@{
	static void flash(bool enable,D3D::Vec4 &color);
	static void fog(float dist, D3D::Vec4 *color);
	static TCHAR *getModes();
	static void getScreenshot(D3D::Vec4_byte* buf);
	static void setBrightness(float brightness);
	static TextureCache& getTextureCache();
	//@}
};